// adapted from Original https://www.shadertoy.com/view/XlXXW8
// Night road created by Maurogik
// Licence CC0
// Adapted, trivialy, for use in VGHD player
/////////////////////////////////////////////
uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels
//#define iGlobalTime u_Elapsed* 0.3141592
//90Â° FOV
#define FOCAL_LENGTH 1.0 
#define iTime u_Elapsed
#define iResolution u_WindowSize
#define iMouse vec4(0.0,0.0,0.0,0.0)
#define MOVE_CAMERA 0.0
#define MOVE_SPEED 1.0
#define FX_SPEED 1000.0
#define MAX_STEPS 512
#define MIN_DIST 0.01
#define MAX_DIST 110.0
#define FOG_STR 1.0
#define GRADIENT_OFFSET 0.1
#define AMBIANT_LIGHT 0.05
#define Z_REPEAT 10.0

uniform sampler2D iChannel0;


//scene data
vec3 projectionUp;
vec3 projectionRight;
vec3 projectionForward;
vec3 projectionCenter;
vec3 cameraOffset;
vec3 lightPos;
vec3 skyColor;
vec2 fragUV;

#ifdef GL_ES
precision mediump float;

#endif
vec4 texture2D_Fract(sampler2D sampler,vec2 P)
 { return texture2D(sampler,fract(P));
 }


float sqrLen(vec3 vec)
{
 	return vec.x * vec.x + vec.y * vec.y + vec.z * vec.z;
}

vec3 lerp(vec3 from, vec3 to, float progress)
{
    return from * (1.0 - progress) + to * progress;
}

vec2 pixelToNormalizedspace(vec2 pixel)
{
    vec2 res;
    res.x = pixel.x * 2.0 / iResolution.x - 1.0;
    res.y = pixel.y * 2.0 / iResolution.y - 1.0;
    res.y *= iResolution.y / iResolution.x;//correct aspect ratio
    return res;
}

vec3 fragCoordToProjectionPlane(vec2 fragUv)
{
	vec2 uv = pixelToNormalizedspace(fragUv);
    return projectionCenter + projectionRight * uv.x + projectionUp * uv.y;
}

//project a direction onto the original (without rotation) projection plane
//to get world space uv
vec2 dirToWorldSpacedUV(vec3 dir)
{
    dir= normalize(dir);
    vec3 projRight 	= vec3(1.0, 0.0, 0.0);
    vec3 projUp 	= vec3(0.0, 1.0, 0.0);
    vec3 projForw	= vec3(0.0, 0.0, 1.0);

    float u = dot(dir, projRight) / dot(projRight, projRight);
    float v = dot(dir, projUp) / dot(projUp, projUp);
    return vec2(u, v);
}

vec2 worldToProjectionPlane(vec3 point)
{
    vec4 camStart = vec4(projectionCenter + cameraOffset, 1.0);
    vec4 camToPoint = vec4(normalize(point - camStart.xyz), 0.0);
    vec4 projectionPlane = vec4(projectionForward, -FOCAL_LENGTH);
    float intersectDist = dot(projectionPlane, camStart) / dot(projectionPlane, camToPoint);
    vec3 pointOnPlane = camStart.xyz + camToPoint.xyz * intersectDist;
    float u = dot(pointOnPlane, projectionRight) / dot(projectionRight, projectionRight);
    float v = dot(pointOnPlane, projectionUp) / dot(projectionUp, projectionUp);
    return vec2(u, v);
}

//Create a shape by removing shape2 from shape1
float subtractField(float df1, float df2)
{
    return max(df1, -df2);
}

//get distance to a cylinder
float cylinderDist(vec3 center, vec2 dimension, vec3 point)
{
  	vec3 p = point - center;   
  	vec2 d = abs(vec2(length(p.xz), p.y)) - dimension;
  	return min(max(d.x, d.y), 0.0) + length(max(d, 0.0));
}

//get distance to an AA box
float AABox(vec3 pmin, vec3 pmax, vec3 point)
{
	float dx = max(0.0, max(pmin.x - point.x, point.x - pmax.x));
    float dy = max(0.0, max(pmin.y - point.y, point.y - pmax.y));
    float dz = max(0.0, max(pmin.z - point.z, point.z - pmax.z));
    return sqrt(dx*dx + dy*dy + dz*dz);    
}

//get distance to surface of a sphere
float sphereDist(vec4 sphere, vec3 point)
{
    return length(point - sphere.xyz) - sphere.w;
}

//get distance to surface of a plane
float planeDist(vec4 plane, vec3 point)
{
    
	return abs(dot(plane, vec4(point, 1.0)));
}


//Pack a plane as a [normal, distToOrigin] vector
vec4 createPlane(vec3 normal, float dist)
{
	return vec4(normalize(normal), dist);
}

//Pack a sphere as a [center, radius] vector
vec4 createSphere(vec3 pos, float radius)
{
 	return vec4(pos, radius);   
}

vec3 cylinderPos	= vec3(3.0, 1.0, 3.0);
vec2 cylinderSize	= vec2(0.1, 2.0);
vec4 plane;
vec4 sphere;
vec3 roadPt1  		= vec3(1.25, -0.95, 0.0) - vec3(2.5, 0.05, 20.0) * 0.5;
vec3 roadPt2		= vec3(1.25, -0.95, 0.0) + vec3(2.5, 0.05, 20.0) * 0.5;
vec3 groundColor 	= vec3(0.3);
vec3 poleColor		= vec3(0.25);
vec3 lampColor 		= vec3(5.0, 5.0, 5.0);
vec3 roadColor		= vec3(0.1);

float calcColorWeight(float dist, float minDist)
{
    return clamp(1.0 - (dist / minDist - 1.0), 0.0, 1.0);
}

vec3 domainMod(vec3 point){
    vec3 moddedPoint	= point;
    moddedPoint.x 		= abs(point.x); //symmetry
    moddedPoint.z 		= mod(point.z, Z_REPEAT);
    return moddedPoint;
}

//get distance and color to nearest surface
vec4 colorDistanceField(vec3 point)
{   
    vec3 moddedPoint	= domainMod(point);
    
    float roadDist		= AABox(roadPt1, roadPt2, moddedPoint);
    float distSphere 	= sphereDist(sphere, moddedPoint);
    float distCylinder 	= cylinderDist(cylinderPos, cylinderSize, moddedPoint);
    float distPlane		= planeDist(plane, point);
    float minDist		= MAX_DIST;
    minDist 			= min(distSphere, distCylinder);    
    minDist 			= min(minDist, distPlane);
    minDist 			= min(minDist, roadDist);
        
    //blend colors
    vec3 color = vec3(0.0);
    color += calcColorWeight(roadDist, minDist) 	* roadColor;
    color += calcColorWeight(distSphere, minDist) 	* lampColor;
    color += calcColorWeight(distCylinder, minDist) * poleColor;
    color += calcColorWeight(distPlane, minDist) 	* groundColor;
    
	return vec4(color, minDist);
}

//get distance to nearest surface
float distanceField(vec3 point)
{   
    vec3 moddedPoint = domainMod(point);
    
    float roadDist		= AABox(roadPt1, roadPt2, moddedPoint);
    float distSphere 	= sphereDist(sphere, moddedPoint);
    float distCylinder 	= cylinderDist(cylinderPos, cylinderSize, moddedPoint);
    float distPlane		= planeDist(plane, point);
    float minDist		= MAX_DIST;
    minDist 			= min(distSphere, distCylinder);    
    minDist 			= min(minDist, distPlane);
    minDist 			= min(minDist, roadDist);
   	return minDist;
}

//get uv for road teture from a world point on a road tile
vec2 pointToRoadUV(vec3 point){ 
    float y =  mod(point.z * 2.0, Z_REPEAT) / Z_REPEAT;
    float x = (point.x - roadPt1.x) / (roadPt2.x - roadPt1.x);
    return vec2(x, y);
}

//raymarch with atmospheric accumulation of color at sampling points
vec4 rmAtmosphere(vec3 rayStart, vec3 rayDir, float minDist)
{
    float totalDist = minDist;
    float dist;
    vec3 color = vec3(0.0);
    float stepBias;
    vec3 atmColor;
    vec4 dfRes;
    
    for(int i = 0; i < MAX_STEPS; ++i)
    {   
        stepBias = max(0.5, float(i) / float(MAX_STEPS));//to avoid artefacts
        dfRes = colorDistanceField(rayStart + rayDir * totalDist);
        dist = dfRes.w;        
        totalDist += dist * stepBias;
        
        if(dist <= MIN_DIST * stepBias)
        {
            atmColor = dfRes.xyz;
            color = lerp(color, skyColor, pow(totalDist / MAX_DIST, 1.0 / FOG_STR));            
        	return vec4((color + atmColor), totalDist);   
        }
        
        if(totalDist >= MAX_DIST)
        {
            return vec4(color + skyColor, MAX_DIST);  
        }                       
        //color accumulation while raymarching to add emission FX
        atmColor = dfRes.xyz;
        color += (clamp(atmColor - vec3(1.0), vec3(0.0), vec3(3.0)) * max(1.5 - dist, 0.0)) / (totalDist * totalDist);    
    }      
        
    return vec4(color + skyColor, totalDist);    
}

//simple distance raymarching
float rmDist(vec3 rayStart, vec3 rayDir, float minDist)
{
    float totalDist = minDist;
    float dist;
    float stepBias;
    
    for(int i = 0; i < MAX_STEPS; ++i)
    {   
        stepBias = 1.0;
        dist = distanceField(rayStart + rayDir * totalDist);
        totalDist += dist * stepBias;
        if(dist <= MIN_DIST)
        {
            return totalDist;
        }
        
        if(totalDist >= MAX_DIST)
        {
            return MAX_DIST;  
        }        
    }      
        
    return totalDist;    
}
void main( void )
//void mainImage( out vec4 fragColor, in vec2 fragCoord )
//void mainImage( out vec4 fragColor, in vec2 fragCoord )
{    
    plane	 		= createPlane(vec3(0.0, 1.0, 0.0), 1.0);
	sphere 		= createSphere(vec3(3.0, 3.0, 3.0), 0.3);
    
    fragUV = gl_FragCoord.xy;      

    //Do camera setup from mouse coord
    projectionCenter 	= vec3(0.0, 0.2, mod(MOVE_SPEED * iTime, 10.0));
    projectionForward 	= vec3(0.0, 0.0, 1.0);
    projectionUp 		= vec3(0.0, 1.0, 0.0);
    projectionRight 	= vec3(1.0, 0.0, 0.0);        
    vec3 mouseProj		= (fragCoordToProjectionPlane(iMouse.xy));
    cameraOffset		= normalize(vec3(-mouseProj.x * MOVE_CAMERA, -mouseProj.y * MOVE_CAMERA, -1.0)) * FOCAL_LENGTH;
    
    //reproject the camera plane
    projectionForward 	= normalize(-cameraOffset);
    projectionUp 		= vec3(0.0, 1.0, 0.0);
    projectionRight 	= cross(projectionUp, projectionForward);
    projectionUp 		= cross(projectionForward, projectionRight);  
    
    lightPos			= projectionCenter + cameraOffset + vec3(0.0, 3.0, 1.5);  
            
    //setup ray
    vec3 rayPos 				= projectionCenter + cameraOffset;
    vec3 pointOnProjectionPlane = fragCoordToProjectionPlane(gl_FragCoord.xy);
    vec3 rayDirection 			= normalize(pointOnProjectionPlane - rayPos);
    
    //sky and stars
    vec2 skyUV 	= dirToWorldSpacedUV(rayDirection) * 1.5;
    vec3 stars 	= texture(iChannel0, skyUV).xyz;
    stars      	*= pow(sqrLen(stars) * 0.45, 10.0) * 0.5;
    float skyDarkness	= pow(abs(rayDirection.y) * 1.5, 0.5);
    skyColor 	= lerp(
          vec3(0.1, 0.1, 0.3)
        , vec3(0.005, 0.005, 0.01) + stars * skyDarkness
        , skyDarkness);    
    
    //march ray
    vec4 rayMarchResult 	= rmAtmosphere(rayPos, rayDirection, 0.0);    
    float dist 				= rayMarchResult.w;
    float distAttenuation 	= pow(1.0 - dist/MAX_DIST * 0.8, 1.0);   
    vec4 surfaceColor 		= vec4(rayMarchResult.xyz, 1.0);

    if(dist < MAX_DIST && sqrLen(rayMarchResult.xyz) < 2.0)//to avoid shadowing the sky
    {
        vec3 endPoint 		= rayPos + rayDirection * (dist);
        vec3 pointToLight 	= normalize(lightPos - endPoint);
        
        //add headlights
        vec3 headlightPoint = endPoint;
        headlightPoint.x 	= abs(headlightPoint.x * 3.5);
        vec3 headlightDiff 	= projectionCenter + vec3(4.0, -0.5, -1.5) - headlightPoint; 
        headlightDiff.z 	*= 0.75;
        float headlight 	= clamp(40.0 / sqrLen(headlightDiff), 0.0, 100.0);        
        
        //normal computation
        float g1 = distanceField(endPoint + vec3(GRADIENT_OFFSET, 0.0, 0.0))
            - distanceField(endPoint + vec3(-GRADIENT_OFFSET, 0.0, 0.0));
        float g2 = distanceField(endPoint + vec3(0.0, GRADIENT_OFFSET, 0.0))
            - distanceField(endPoint + vec3(0.0, -GRADIENT_OFFSET, 0.0));
        float g3 = distanceField(endPoint + vec3(0.0, 0.0, GRADIENT_OFFSET))
            - distanceField(endPoint + vec3(0.0, 0.0, -GRADIENT_OFFSET));
        vec3 normal = normalize(vec3(g1, g2, g3));
        
        //road normal from texture
        vec2 normalUV = pointToRoadUV(endPoint);
        normal += texture(iChannel0, normalUV).xyz 
            * (1.0 - clamp(AABox(roadPt1, roadPt2, domainMod(endPoint)) * 5.0, 0.0, 1.0)) * 0.5;
        normal = normalize(normal);
        
        //dist to lamps
        float lampLight = min(pow(22.0 / sqrLen(domainMod(endPoint + normal * 0.1) - sphere.xyz), 20.0), 3.0);
        lampLight /= 3.0;
        
        //lighting    
        float spec = pow(clamp(dot(normalize(-rayDirection + pointToLight), normal), 0.0, 1.0), 16.0);
        float ambiant = (AMBIANT_LIGHT * distAttenuation + lampLight) * pow(distAttenuation, 2.0);
        surfaceColor = surfaceColor * ambiant 
            + surfaceColor * max(0.0, dot(normal, pointToLight) * headlight)
            + vec4(0.8) * spec * headlight;
        
	    //check for shadow casting        
        float shadowDist 	= rmDist(endPoint + normal * 0.01, pointToLight, 0.0);    
    	float shadow			= 1.0 - (shadowDist / MAX_DIST);
        shadow					= clamp(shadow - lampLight * 0.3, 0.0, 1.0);
    	float lightIntensity 	= 1.0 - pow(shadow, 4.0);
 	   	surfaceColor 			*= lightIntensity;     
    }
    
    gl_FragColor = surfaceColor;
}
